// === Elements ===
const gridItems = document.querySelectorAll(".grid-item");
const logo = document.querySelector(".logo");
const body = document.body;

const themeSelect = document.getElementById("theme-select");
const fontSelect = document.getElementById("font-select");
const volumeSlider = document.getElementById("volume-slider");
const textureToggle = document.getElementById("texture-toggle");
const siteTitle = document.getElementById("site-title");

const settingsBtn = document.getElementById("settings");
const settingsPanel = document.getElementById("settings-panel");
const closeSettingsBtn = document.getElementById("close-settings");

// === State ===
let currentAudio = null;
let currentItem = null;

// Get saved volume or default
let globalVolume = parseFloat(localStorage.getItem("volume")) || 0.7;

// ===== AUDIO PLAYER =====
gridItems.forEach(item => {
  const sound = item.dataset.sound;
  const audio = new Audio(sound);

  // Apply saved volume
  audio.volume = globalVolume;

  item.addEventListener("click", () => {
    // Toggle pause/play if same item clicked
    if (currentItem === item) {
      if (audio.paused) {
        audio.play();
        item.classList.add("playing");
        item.classList.remove("paused");
        logo.classList.add("playing-logo");
      } else {
        audio.pause();
        item.classList.remove("playing");
        item.classList.add("paused");
        logo.classList.remove("playing-logo");
      }
      return;
    }

    // Pause previous
    if (currentAudio) {
      currentAudio.pause();
      currentItem.classList.remove("playing");
      currentItem.classList.add("paused");
    }

    // Play new
    audio.volume = globalVolume; // ensure volume is correct
    audio.play();
    item.classList.add("playing");
    item.classList.remove("paused");
    logo.classList.add("playing-logo");

    currentAudio = audio;
    currentItem = item;

    localStorage.setItem("lastSound", sound);

    audio.addEventListener("ended", () => {
      item.classList.remove("playing");
      item.classList.add("paused");
      logo.classList.remove("playing-logo");
    });
  });
});

// ===== SETTINGS PANEL =====
settingsBtn?.addEventListener("click", () => {
  settingsPanel.classList.toggle("hidden");
});
closeSettingsBtn?.addEventListener("click", () => {
  settingsPanel.classList.add("hidden");
});

// ===== SETTINGS CONTROLS =====

// Theme
themeSelect?.addEventListener("change", e => {
  const theme = e.target.value;
  if (theme === "none") {
    body.className = "";
    body.style.backgroundImage = "url('main bg.jpg')";
    body.style.color = "#fff";
  } else {
    body.className = theme;
    body.style.backgroundImage = "";
  }
  localStorage.setItem("theme", theme);
});

// Font
fontSelect?.addEventListener("change", e => {
  const font = e.target.value;
  body.style.fontFamily = font;
  localStorage.setItem("font", font);
});

// Volume
volumeSlider?.addEventListener("input", e => {
  globalVolume = parseFloat(e.target.value);

  // Update currently playing audio
  if (currentAudio) currentAudio.volume = globalVolume;

  // Save to localStorage
  localStorage.setItem("volume", globalVolume);
});

// Texture
textureToggle?.addEventListener("change", e => {
  const showTexture = e.target.checked;
  body.style.setProperty("--texture-opacity", showTexture ? "1" : "0");
  body.classList.toggle("no-texture", !showTexture);
  localStorage.setItem("texture", showTexture);
});

// Site title
siteTitle?.addEventListener("input", () => {
  localStorage.setItem("title", siteTitle.textContent);
});

// ===== RESTORE LOCAL STORAGE ON LOAD =====
window.addEventListener("DOMContentLoaded", () => {
  // Theme
  const savedTheme = localStorage.getItem("theme");
  if (savedTheme && savedTheme !== "none") {
    body.className = savedTheme;
    themeSelect.value = savedTheme;
  } else {
    body.className = "";
    themeSelect.value = "none";
    body.style.backgroundImage = "url('main bg.jpg')";
    body.style.color = "#fff";
  }

  // Font
  const savedFont = localStorage.getItem("font");
  if (savedFont) {
    body.style.fontFamily = savedFont;
    fontSelect.value = savedFont;
  }

  // Volume
  const savedVol = localStorage.getItem("volume");
  if (savedVol !== null) {
    globalVolume = parseFloat(savedVol);
    volumeSlider.value = globalVolume;
  }

  // Texture
  const savedTexture = localStorage.getItem("texture");
  if (savedTexture === "false") {
    if (textureToggle) textureToggle.checked = false;
    body.style.setProperty("--texture-opacity", "0");
    body.classList.add("no-texture");
  } else if (savedTexture === "true") {
    if (textureToggle) textureToggle.checked = true;
  }

  // Site title
  const savedTitle = localStorage.getItem("title");
  if (savedTitle) {
    siteTitle.textContent = savedTitle;
  }
});
